package com.ctrip.framework.apollo.adminservice.controller;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.List;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.test.util.ReflectionTestUtils;

import com.ctrip.framework.apollo.biz.entity.App;
import com.ctrip.framework.apollo.biz.service.AdminService;
import com.ctrip.framework.apollo.biz.service.AppService;
import com.ctrip.framework.apollo.core.dto.AppDTO;
import com.ctrip.framework.apollo.core.exception.NotFoundException;
import com.ctrip.framework.apollo.core.exception.ServiceException;

@RunWith(MockitoJUnitRunner.class)
public class ControllerExceptionTest {

  private AppController appController;

  @Mock
  private AppService appService;

  @Mock
  private AdminService adminService;

  @Before
  public void setUp() {
    appController = new AppController();
    ReflectionTestUtils.setField(appController, "appService", appService);
    ReflectionTestUtils.setField(appController, "adminService", adminService);
  }

  @Test(expected = NotFoundException.class)
  public void testFindNotExists() {
    when(appService.findOne(any(String.class))).thenReturn(null);
    appController.get("unexist");
  }

  @Test(expected = NotFoundException.class)
  public void testDeleteNotExists() {
    when(appService.findOne(any(String.class))).thenReturn(null);
    appController.delete("unexist", null);
  }

  @Test
  public void testFindEmpty() {
    when(appService.findAll(any(Pageable.class))).thenReturn(new ArrayList<App>());
    Pageable pageable = new PageRequest(0, 10);
    List<AppDTO> appDTOs = appController.find(null, pageable);
    Assert.assertNotNull(appDTOs);
    Assert.assertEquals(0, appDTOs.size());

    appDTOs = appController.find("", pageable);
    Assert.assertNotNull(appDTOs);
    Assert.assertEquals(0, appDTOs.size());
  }

  @Test
  public void testFindByName() {
    Pageable pageable = new PageRequest(0, 10);
    List<AppDTO> appDTOs = appController.find("unexist", pageable);
    Assert.assertNotNull(appDTOs);
    Assert.assertEquals(0, appDTOs.size());
  }

  @Test(expected = ServiceException.class)
  public void createFailed() {
    AppDTO dto = generateSampleDTOData();

    when(appService.findOne(any(String.class))).thenReturn(null);
    when(adminService.createNewApp(any(App.class)))
        .thenThrow(new ServiceException("create app failed"));

    UserDetails user = new User("user", "", new ArrayList<GrantedAuthority>());
    appController.createOrUpdate(dto, user);
  }

  private AppDTO generateSampleDTOData() {
    AppDTO dto = new AppDTO();
    dto.setAppId("someAppId");
    dto.setName("someName");
    dto.setOwnerName("someOwner");
    dto.setOwnerEmail("someOwner@ctrip.com");
    return dto;
  }
}
