package com.ctrip.framework.apollo.portal.auth;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.ctrip.framework.apollo.portal.entity.po.UserInfo;
import com.ctrip.framework.apollo.portal.service.ServerConfigService;
import com.ctrip.framework.apollo.portal.service.UserService;

import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author Jason Song(song_s@ctrip.com)
 */
public class CtripUserService implements UserService {
  private ServerConfigService serverConfigService;
  private RestTemplate restTemplate;
  private List<String> searchUserMatchFields;
  private ParameterizedTypeReference<Map<String, List<UserServiceResponse>>> responseType;

  public CtripUserService(ServerConfigService serverConfigService) {
    this.serverConfigService = serverConfigService;
    this.restTemplate = new RestTemplate(clientHttpRequestFactory());
    this.searchUserMatchFields =
        Lists.newArrayList("empcode", "empaccount", "displayname", "c_name", "pinyin");
    this.responseType = new ParameterizedTypeReference<Map<String, List<UserServiceResponse>>>() {
    };
  }

  private ClientHttpRequestFactory clientHttpRequestFactory() {
    SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
    factory.setConnectTimeout(getConnectTimeout());
    factory.setReadTimeout(getReadTimeout());
    return factory;
  }

  private int getConnectTimeout() {
    String connectTimeout = serverConfigService.getValue("api.connectTimeout", "3000");

    return Integer.parseInt(connectTimeout);
  }

  private int getReadTimeout() {
    String readTimeout = serverConfigService.getValue("api.readTimeout", "3000");

    return Integer.parseInt(readTimeout);
  }

  @Override
  public List<UserInfo> searchUsers(String keyword, int offset, int limit) {
    UserServiceRequest request = assembleSearchUserRequest(keyword, offset, limit);

    HttpEntity<UserServiceRequest> entity = new HttpEntity<>(request);
    ResponseEntity<Map<String, List<UserServiceResponse>>> response =
        restTemplate.exchange(getUserServiceUrl(), HttpMethod.POST, entity, responseType);

    if (!response.getBody().containsKey("result")) {
      return Collections.emptyList();
    }

    List<UserInfo> result = Lists.newArrayList();
    result.addAll(
        response.getBody().get("result").stream().map(this::transformUserServiceResponseToUserInfo)
            .collect(Collectors.toList()));

    return result;
  }

  @Override
  public UserInfo findByUserId(String userId) {
    List<UserInfo> userInfoList = this.findByUserIds(Lists.newArrayList(userId));
    if (CollectionUtils.isEmpty(userInfoList)) {
      return null;
    }
    return userInfoList.get(0);
  }

  public List<UserInfo> findByUserIds(List<String> userIds) {
    UserServiceRequest request = assembleFindUserRequest(Lists.newArrayList(userIds));

    HttpEntity<UserServiceRequest> entity = new HttpEntity<>(request);
    ResponseEntity<Map<String, List<UserServiceResponse>>> response =
        restTemplate.exchange(getUserServiceUrl(), HttpMethod.POST, entity, responseType);

    if (!response.getBody().containsKey("result")) {
      return Collections.emptyList();
    }

    List<UserInfo> result = Lists.newArrayList();
    result.addAll(
        response.getBody().get("result").stream().map(this::transformUserServiceResponseToUserInfo)
            .collect(Collectors.toList()));

    return result;
  }

  private UserInfo transformUserServiceResponseToUserInfo(UserServiceResponse userServiceResponse) {
    UserInfo userInfo = new UserInfo();
    userInfo.setUserId(userServiceResponse.getEmpaccount());
    userInfo.setName(userServiceResponse.getDisplayname());
    userInfo.setEmail(userServiceResponse.getEmail());
    return userInfo;
  }

  UserServiceRequest assembleSearchUserRequest(String keyword, int offset, int limit) {
    Map<String, Object> query = Maps.newHashMap();
    Map<String, Object> multiMatchMap = Maps.newHashMap();
    multiMatchMap.put("fields", searchUserMatchFields);
    multiMatchMap.put("operator", "and");
    multiMatchMap.put("query", keyword);
    multiMatchMap.put("type", "best_fields");
    query.put("multi_match", multiMatchMap);

    return assembleUserServiceRequest(query, offset, limit);
  }

  UserServiceRequest assembleFindUserRequest(List<String> userIds) {
    Map<String, Object>
        query =
        ImmutableMap.of("filtered", ImmutableMap
            .of("filter", ImmutableMap.of("terms", ImmutableMap.of("empaccount", userIds))));

    return assembleUserServiceRequest(query, 0, userIds.size());
  }

  private UserServiceRequest assembleUserServiceRequest(Map<String, Object> query, int offset,
                                                        int limit) {
    UserServiceRequest request = new UserServiceRequest();
    request.setAccess_token(getUserServiceAccessToken());
    request.setType("emloyee");

    UserServiceRequestBody requestBody = new UserServiceRequestBody();
    requestBody.setIndexAlias("itdb_emloyee");
    request.setRequest_body(requestBody);

    Map<String, Object> queryJson = Maps.newHashMap();
    requestBody.setQueryJson(queryJson);

    queryJson.put("query", query);

    queryJson.put("from", offset);
    queryJson.put("size", limit);

    return request;
  }


  private String getUserServiceUrl() {
    return serverConfigService.getValue("userService.url");
  }

  private String getUserServiceAccessToken() {
    return serverConfigService.getValue("userService.accessToken");
  }

  static class UserServiceRequest {
    private String access_token;
    private String type;
    private UserServiceRequestBody request_body;

    public String getAccess_token() {
      return access_token;
    }

    public void setAccess_token(String access_token) {
      this.access_token = access_token;
    }

    public String getType() {
      return type;
    }

    public void setType(String type) {
      this.type = type;
    }

    public UserServiceRequestBody getRequest_body() {
      return request_body;
    }

    public void setRequest_body(
        UserServiceRequestBody request_body) {
      this.request_body = request_body;
    }
  }

  static class UserServiceRequestBody {
    private String indexAlias;
    private Map<String, Object> queryJson;

    public String getIndexAlias() {
      return indexAlias;
    }

    public void setIndexAlias(String indexAlias) {
      this.indexAlias = indexAlias;
    }

    public Map<String, Object> getQueryJson() {
      return queryJson;
    }

    public void setQueryJson(Map<String, Object> queryJson) {
      this.queryJson = queryJson;
    }
  }

  static class UserServiceResponse {
    private String empaccount;
    private String displayname;
    private String email;

    public String getEmpaccount() {
      return empaccount;
    }

    public void setEmpaccount(String empaccount) {
      this.empaccount = empaccount;
    }

    public String getDisplayname() {
      return displayname;
    }

    public void setDisplayname(String displayname) {
      this.displayname = displayname;
    }

    public String getEmail() {
      return email;
    }

    public void setEmail(String email) {
      this.email = email;
    }
  }

}
